"""Base repository to wrap TinyDB CRUD operations with ID auto‑generation."""

from __future__ import annotations

from typing import Any, Dict, List, Optional, Type
from tinydb import TinyDB, Query


class TinyDBRepository:
    """A generic repository providing CRUD operations for TinyDB tables."""

    def __init__(self, db_path: str, table_name: str, model_class: Type[Any]):
        self.db_path = db_path
        self.db = TinyDB(db_path)
        self.table = self.db.table(table_name)
        self.model_class = model_class

    def _generate_id(self) -> int:
        """Generate the next integer ID by inspecting existing rows."""
        items = self.table.all()
        if not items:
            return 1
        max_id = max(item.get("id", 0) for item in items)
        return max_id + 1

    def insert(self, data: Dict[str, Any]) -> int:
        """Insert a new record and return its assigned ID."""
        if "id" not in data or data["id"] is None:
            data["id"] = self._generate_id()
        self.table.insert(data)
        return data["id"]

    def get(self, item_id: int) -> Optional[Dict[str, Any]]:
        """Retrieve a record by its ID."""
        result = self.table.get(Query().id == item_id)
        return result

    def all(self) -> List[Dict[str, Any]]:
        return self.table.all()

    def update(self, item_id: int, updates: Dict[str, Any]) -> bool:
        """Update an existing record by ID."""
        result = self.table.update(updates, Query().id == item_id)
        return bool(result)

    def delete(self, item_id: int) -> bool:
        """Remove a record by ID."""
        result = self.table.remove(Query().id == item_id)
        return bool(result)

    def filter(self, **conditions: Any) -> List[Dict[str, Any]]:
        """Return all records matching the provided field conditions."""
        query = None
        for field, value in conditions.items():
            q = Query()[field] == value
            query = q if query is None else (query & q)
        if query is None:
            return self.all()
        return self.table.search(query)